'use strict';

/* --------------------------------------------------------------
 info_box.js 2019-07-12
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2019 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

jse.libs.info_box = jse.libs.info_box || {};

/**
 * ## Info Box Messages Library
 *
 * This module provides an API to the new admin layout pages info box.
 *
 * @module Admin/Libs/info_box
 * @exports jse.libs.info_box
 */
(function (exports) {

    'use strict';

    // Admin info box element selector.

    var infoboxSelector = '.info-box';

    /**
     * Performs an ajax request to the server.
     *
     * @param {String} action URL action part.
     * @param {String} method HTTP request method.
     * @param {Object} data   Request data.
     *
     * @return {Deferred|Promise}
     *
     * @private
     */
    function _performRequest(action, method, data) {
        var URL_BASE = 'admin.php?do=AdminInfoBoxAjax';

        // AJAX request options.
        var ajaxOptions = {
            url: URL_BASE + action,
            dataType: 'json',
            data: data,
            method: method
        };

        // Returns deferred object.
        return $.ajax(ajaxOptions);
    }

    // Message status.
    exports.STATUS_NEW = 'new';
    exports.STATUS_READ = 'read';
    exports.STATUS_HIDDEN = 'hidden';
    exports.STATUS_DELETED = 'deleted';

    // Message types.
    exports.TYPE_INFO = 'info';
    exports.TYPE_WARNING = 'warning';
    exports.TYPE_SUCCESS = 'success';

    // Message visibility.
    exports.VISIBILITY_ALWAYS_ON = 'alwayson';
    exports.VISIBILITY_HIDEABLE = 'hideable';
    exports.VISIBILITY_REMOVABLE = 'removable';

    // Admin action success message identifier prefix.
    exports.SUCCESS_MSG_IDENTIFIER_PREFIX = 'adminActionSuccess-';

    /**
     * Returns the messages from the server (visible only).
     *
     * @return {Promise}
     */
    exports.getMessages = function () {
        return _performRequest('/GetAllMessages', 'GET');
    };

    /**
     * Sets the status of a message.
     *
     * @param {Number} id Message ID.
     * @param {String} status Message status to set ('new', 'read', 'hidden', 'deleted').
     *
     * @return {Promise}
     */
    exports.setStatus = function (id, status) {
        // Valid message status.
        var validStatus = [exports.STATUS_NEW, exports.STATUS_READ, exports.STATUS_HIDDEN, exports.STATUS_DELETED];

        // Check arguments.
        if (!id || !status) {
            throw new Error('Missing ID or status');
        } else if (validStatus.indexOf(status) === -1) {
            throw new Error('Invalid status provided');
        }

        return _performRequest('/SetMessageStatus', 'GET', { id: id, status: status });
    };

    /**
     * Deletes a message.
     *
     * @param {Number} id Message ID.
     *
     * @return {Promise}
     */
    exports.deleteById = function (id) {
        if (!id) {
            throw new Error('Missing ID');
        }

        return _performRequest('/DeleteById', 'GET', { id: id });
    };

    /**
     * Deletes a message by source.
     *
     * @param {String} source Message source.
     *
     * @return {Promise}
     */
    exports.deleteBySource = function (source) {
        if (!source) {
            throw new Error('Missing source');
        }

        return _performRequest('/DeleteBySource', 'GET', { source: source });
    };

    /**
     * Deletes a messages by the identifier.
     *
     * @param {String} identifier Message identifier.
     *
     * @return {Promise}
     */
    exports.deleteByIdentifier = function (identifier) {
        if (!identifier) {
            throw new Error('Missing identifier');
        }

        return _performRequest('/DeleteByIdentifier', 'GET', { identifier: identifier });
    };

    /**
     * Reactivates the messages.
     * @return {Promise}
     * @static
     */
    exports.Reactivates = function () {
        return _performRequest('/ReactivateMessages', 'GET');
    };

    /**
     * Saves a new message.
     *
     * @param {Object} message The new message to save.
     *
     * @example
     * jse.libs.info_box.addMessage({
     *   source: 'ajax',
     *   identifier: 'asdas',
     *   status: 'new',
     *   type: 'success',
     *   visibility: 'removable',
     *   customerId: 0,
     *   headline: 'My Headline',
     *   buttonLabel: 'asdas',
     *   buttonLink: 'http://google.com', // optional
     *	 buttonLink: 'customers.php', // optional
     *   message: 'Hallo!',
     * });
     *
     * @return {Promise}
     */
    exports.addMessage = function (message) {
        if (!message) {
            throw new Error('Missing message object');
        } else if (!message.source) {
            throw new Error('Missing source');
        } else if (!message.identifier) {
            message.identifier = 'generated-' + Date.now();
        } else if (!message.status || [exports.STATUS_NEW, exports.STATUS_READ, exports.STATUS_HIDDEN, exports.STATUS_DELETED].indexOf(message.status) === -1) {
            throw new Error('Missing or invalid status');
        } else if (!message.type || [exports.TYPE_INFO, exports.TYPE_WARNING, exports.TYPE_SUCCESS].indexOf(message.type) === -1) {
            throw new Error('Missing or invalid type');
        } else if (!message.visibility || [exports.VISIBILITY_ALWAYS_ON, exports.VISIBILITY_HIDEABLE, exports.VISIBILITY_REMOVABLE].indexOf(message.visibility) === -1) {
            throw new Error('Missing or invalid visibility');
        } else if (typeof message.customerId === 'undefined') {
            throw new Error('Missing customer ID');
        } else if (!message.message) {
            throw new Error('Missing message');
        } else if (!message.headline) {
            throw new Error('Missing headline');
        } else if (!message.message) {
            throw new Error('Missing message');
        }

        return _performRequest('/AddMessage', 'GET', message);
    };

    /**
     * Adds a success message to the admin info box.
     *
     * @param {String} [message] Message to show.
     * @param {Boolean} [skipRefresh = false] Refresh the admin info box to show the message?
     */
    exports.addSuccessMessage = function (message) {
        var skipRefresh = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;

        // Add message.
        var request = _performRequest('/AddSuccessMessage', 'GET', message ? { message: message } : {});

        // Optionally refresh the admin info box to show the message.
        if (!skipRefresh) {
            request.done(function () {
                return $(infoboxSelector).trigger('refresh:messages');
            });
        }
    };
})(jse.libs.info_box);
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
